import os
import subprocess
import time
import random
import string
import threading
import sys
import tkinter as tk
from tkinter import ttk, messagebox, scrolledtext, filedialog, font
import shutil
import urllib.request
import webbrowser
# --- Setup Enforcement ---
if getattr(sys, 'frozen', False):
    CURRENT_DIR = os.path.dirname(sys.executable)
else:
    CURRENT_DIR = os.path.dirname(os.path.abspath(__file__))

SETUP_FLAG = os.path.join(CURRENT_DIR, "setup_done.flag")

if not os.path.exists(SETUP_FLAG):
    root = tk.Tk()
    root.withdraw()
    messagebox.showwarning("Setup Required", 
                           "The system detected that the initial setup has not been completed.\n\n"
                           "Please run 'Click Here Before Start' first to install required dependencies.")
    root.destroy()
    sys.exit(1)

import atw_loader

# --- Configuration & Constants ---
# (CURRENT_DIR is already defined above)

# --- Encrypted Folder Loading ---
print("[*] Accessing secure environment...")
PROTECTED_ATW_PATH = atw_loader.load_atw_server()
if not PROTECTED_ATW_PATH:
    messagebox.showerror("Security Error", "Could not initialize secure environment. File might be missing or corrupted.")
    sys.exit(1)

# Base directory for other assets
BASE_DIR = CURRENT_DIR

# Tor paths relative to protected folder
TOR_EXECUTABLE_PATH = os.path.join(PROTECTED_ATW_PATH, "atw", "atw.exe")
DATA_DIR_PATH = os.path.join(PROTECTED_ATW_PATH, "data", "tor_temp")
GEOIP_FILE = os.path.join(PROTECTED_ATW_PATH, "data", "geoip")
GEOIP6_FILE = os.path.join(PROTECTED_ATW_PATH, "data", "geoip6")

# --- Colors (Phantom Premium Theme) ---
BG_COLOR = "#05080A"       # Deepest Void
SIDEBAR_BG = "#0B0F14"     # Dark Panel
FG_COLOR = "#E6EDF3"       # Bright Text
ACCENT_COLOR = "#00D6B4"   # Cyan/Teal Neon (Phantom Glow)
ACCENT_HOVER = "#00F0C9"   # Brighter Cyan
ALERT_COLOR = "#FF3333"    # Professional Red
SECONDARY_TEXT = "#8B949E" 
BORDER_COLOR = "#20262C"
INPUT_BG = "#0D1117"
INPUT_BORDER = "#30363D"
HIGHLIGHT_COLOR = "#1F6FEB" # Blue for focus

class PhantomProxyBot:
    def __init__(self, root):
        self.root = root
        self.root.title("ALI PHANTOM PROXY [ENTERPRISE EDITION]")
        self.root.geometry("1180x850") # Slightly larger
        self.root.configure(bg=BG_COLOR)
        
        # Variables
        self.tor_process = None
        self.is_running = False
        
        # Inputs
        self.ports_var = tk.StringVar(value="9050") 
        self.gen_limit_var = tk.StringVar(value="100") # Default increased
        self.country_var = tk.StringVar(value="United States (US)")
        self.proxy_mode_var = tk.StringVar(value="sticky") 
        self.proxy_mode_var = tk.StringVar(value="sticky")
        
        # Input Widget References (for Locking)
        self.widgets_to_lock = []
        
        # Styles
        self._configure_styles()
        
        # Layout
        self._build_layout()
        
        self.log("PHANTOM ENGINE INITIALIZED. WAITING FOR COMMAND.")
        self.check_env()

    def _configure_styles(self):
        style = ttk.Style()
        style.theme_use('clam')
        
        # Fonts - Professional & Unique
        # Try 'Bahnschrift', 'Calibri', 'Century Gothic' fallback to 'Segoe UI'
        base_font = ("Bahnschrift", 11) if "Bahnschrift" in font.families() else ("Segoe UI", 11)
        header_font = ("Bahnschrift", 12, "bold") if "Bahnschrift" in font.families() else ("Segoe UI", 12, "bold")
        
        # Base
        style.configure(".", background=BG_COLOR, foreground=FG_COLOR, borderwidth=0)
        
        # Labels
        style.configure("TLabel", background=BG_COLOR, foreground=FG_COLOR, font=base_font)
        style.configure("Sidebar.TLabel", background=SIDEBAR_BG, foreground=SECONDARY_TEXT, font=("Segoe UI Semibold", 10))
        # Status Label
        style.configure("Status.TLabel", font=header_font, background=SIDEBAR_BG)
        
        # Buttons
        style.configure("Primary.TButton", background=ACCENT_COLOR, foreground="#000000", font=header_font, padding=12, borderwidth=0)
        style.map("Primary.TButton", background=[('active', ACCENT_HOVER), ('disabled', '#222222')])
        
        style.configure("Danger.TButton", background=ALERT_COLOR, foreground="#FFFFFF", font=header_font, padding=12, borderwidth=0)
        style.map("Danger.TButton", background=[('active', '#CC0000')])

        style.configure("Ghost.TButton", background="#21262D", foreground=ACCENT_COLOR, font=(base_font[0], 10, "bold"), padding=8)
        style.map("Ghost.TButton", background=[('active', '#30363D')])

        # Radiobutton
        style.configure("TRadiobutton", background=SIDEBAR_BG, foreground=FG_COLOR, font=("Segoe UI", 10))
        style.map("TRadiobutton", background=[('active', SIDEBAR_BG), ('selected', SIDEBAR_BG)])
        
        # Combobox - High Contrast Fix (White BG / Black Text for visibility)
        style.map('TCombobox', 
                  fieldbackground=[('readonly', '#FFFFFF'), ('!disabled', '#FFFFFF')],
                  selectbackground=[('readonly', '#FFFFFF'), ('!disabled', '#FFFFFF')],
                  selectforeground=[('readonly', '#000000'), ('!disabled', '#000000')],
                  foreground=[('readonly', '#000000'), ('!disabled', '#000000')])
                  
        style.configure("TCombobox", 
                        arrowcolor=ACCENT_COLOR, 
                        bordercolor=INPUT_BORDER, 
                        darkcolor='#FFFFFF', 
                        lightcolor='#FFFFFF',
                        background='#FFFFFF',
                        foreground='#000000')
        
        # Force popdown listbox to be White/Black
        self.root.option_add('*TCombobox*Listbox.background', '#FFFFFF')
        self.root.option_add('*TCombobox*Listbox.foreground', '#000000')
        self.root.option_add('*TCombobox*Listbox.selectBackground', ACCENT_COLOR)
        self.root.option_add('*TCombobox*Listbox.selectForeground', '#000000')

    def _build_layout(self):
        # Container
        container = tk.Frame(self.root, bg=BG_COLOR)
        container.pack(fill="both", expand=True)

        # --- Sidebar ---
        sidebar = tk.Frame(container, bg=SIDEBAR_BG, width=350)
        sidebar.pack(side="left", fill="y")
        sidebar.pack_propagate(False)

        # Branding Header
        logo_frame = tk.Frame(sidebar, bg=SIDEBAR_BG, pady=40)
        logo_frame.pack(fill="x")
        
        logo_label = tk.Label(logo_frame, text="AP", bg=SIDEBAR_BG, fg=ACCENT_COLOR, font=("Impact", 48))
        logo_label.pack()
        
        subtitle_label = tk.Label(logo_frame, text="PHANTOM\nPROXY", bg=SIDEBAR_BG, fg=FG_COLOR, font=("Bahnschrift", 16, "bold"), justify="center")
        subtitle_label.pack(pady=(5,0))

        # Status
        self.status_frame = tk.Frame(sidebar, bg=SIDEBAR_BG)
        self.status_frame.pack(fill="x", pady=10)
        self.status_dot = tk.Canvas(self.status_frame, width=16, height=16, bg=SIDEBAR_BG, highlightthickness=0)
        self.status_dot.pack(side="left", padx=(40, 10))
        self.status_light = self.status_dot.create_oval(2, 2, 14, 14, fill="#555", outline="")
        self.status_text = ttk.Label(self.status_frame, text="SYSTEM OFFLINE", style="Status.TLabel", foreground="#777")
        self.status_text.pack(side="left")

        # Controls
        self._add_sidebar_section(sidebar, "SECURITY CONFIGURATION")
        
        # Starting Port
        self.ports_entry = self._add_input(sidebar, "Starting Port (Auto-Increments)", self.ports_var)
        self.widgets_to_lock.append(self.ports_entry)
        
        # Count
        self.limit_entry = self._add_input(sidebar, "Total Proxies (Max 1000)", self.gen_limit_var)
        self.widgets_to_lock.append(self.limit_entry)

        # Country
        ttk.Label(sidebar, text="Exit Node Location", style="Sidebar.TLabel").pack(anchor="w", padx=25, pady=(15, 5))
        countries = [
            "United States (US)", "United Kingdom (GB)", "Canada (CA)", "Germany (DE)", 
            "France (FR)", "Netherlands (NL)", "Australia (AU)", "Japan (JP)", 
            "Pakistan (PK)", "India (IN)", "Singapore (SG)", "Brazil (BR)", 
            "Random (World)"
        ]
        self.country_cb = ttk.Combobox(sidebar, textvariable=self.country_var, values=countries, state="readonly", font=("Bahnschrift", 10))
        self.country_cb.pack(fill="x", padx=25, pady=5)
        self.widgets_to_lock.append(self.country_cb)
        
        # Mode Selection
        tk.Frame(sidebar, height=15, bg=SIDEBAR_BG).pack()
        ttk.Label(sidebar, text="Proxy Behavior Mode", style="Sidebar.TLabel").pack(anchor="w", padx=25, pady=(5,5))
        
        self.rb_sticky = ttk.Radiobutton(sidebar, text="Sticky IP (Lock Session)", variable=self.proxy_mode_var, value="sticky")
        self.rb_sticky.pack(anchor="w", padx=25, pady=2)
        self.widgets_to_lock.append(self.rb_sticky)
        
        self.rb_rot_5 = ttk.Radiobutton(sidebar, text="Rotating IP (5 Minutes)", variable=self.proxy_mode_var, value="rotating_5m")
        self.rb_rot_5.pack(anchor="w", padx=25, pady=2)
        self.widgets_to_lock.append(self.rb_rot_5)

        self.rb_rot_10 = ttk.Radiobutton(sidebar, text="Rotating IP (10 Minutes)", variable=self.proxy_mode_var, value="rotating_10m")
        self.rb_rot_10.pack(anchor="w", padx=25, pady=2)
        self.widgets_to_lock.append(self.rb_rot_10)
        
        # Initial State
        # self.update_mode_ui() -> Removed

        # Actions
        tk.Frame(sidebar, height=30, bg=SIDEBAR_BG).pack()
        self.start_btn = ttk.Button(sidebar, text="ACTIVATE PHANTOM", style="Primary.TButton", command=self.toggle_tor)
        self.start_btn.pack(fill="x", padx=25, pady=8)
        
        self.stop_btn = ttk.Button(sidebar, text="TERMINATE ENGINE", style="Danger.TButton", command=self.stop_tor, state="disabled")
        self.stop_btn.pack(fill="x", padx=25, pady=8)

        # --- Main Content ---
        main = tk.Frame(container, bg=BG_COLOR)
        main.pack(side="right", fill="both", expand=True, padx=40, pady=40)

        # Top Bar
        top_bar = tk.Frame(main, bg=BG_COLOR)
        top_bar.pack(fill="x", pady=(0, 25))
        
        tk.Label(top_bar, text="GENERATED IDENTITIES", bg=BG_COLOR, fg=FG_COLOR, font=("Bahnschrift", 14, "bold")).pack(side="left")
        
        # Gen Controls
        gen_frame = tk.Frame(top_bar, bg=BG_COLOR)
        gen_frame.pack(side="right")
        
        ttk.Button(gen_frame, text="REFRESH LIST", style="Ghost.TButton", command=self.generate_proxies).pack(side="left", padx=8)
        ttk.Button(gen_frame, text="COPY ALL", style="Ghost.TButton", command=self.copy_clipboard).pack(side="left", padx=8)

        # Output
        self.output_area = scrolledtext.ScrolledText(main, bg=INPUT_BG, fg=ACCENT_COLOR, insertbackground=ACCENT_COLOR, 
                                                     font=("Consolas", 11), relief="flat", padx=15, pady=15, borderwidth=1)
        self.output_area.pack(fill="both", expand=True, pady=(0, 25))

        # Logs
        tk.Label(main, text="SYSTEM DIAGNOSTICS", bg=BG_COLOR, fg=SECONDARY_TEXT, font=("Bahnschrift", 10, "bold")).pack(anchor="w")
        self.log_area = scrolledtext.ScrolledText(main, height=10, bg="#05080A", fg="#48525E", 
                                                  font=("Consolas", 9), state="disabled", relief="flat", borderwidth=1)
        self.log_area.pack(fill="x", pady=(10, 0))



    # --- Helpers ---
    def _add_sidebar_section(self, parent, title):
        tk.Frame(parent, height=1, bg=BORDER_COLOR).pack(fill="x", pady=(25, 15))
        ttk.Label(parent, text=title, style="Sidebar.TLabel").pack(anchor="w", padx=25)

    def _add_input(self, parent, label, var):
        ttk.Label(parent, text=label, style="Sidebar.TLabel").pack(anchor="w", padx=25, pady=(10, 5))
        # Use simple entry style
        entry = tk.Entry(parent, textvariable=var, bg=INPUT_BG, fg=FG_COLOR, insertbackground=ACCENT_COLOR, relief="solid", bd=1, font=("Consolas", 11))
        entry.pack(fill="x", padx=25, ipady=8)
        return entry

    def log(self, message):
        timestamp = time.strftime("%H:%M:%S")
        self.log_area.config(state="normal")
        self.log_area.insert("end", f"[{timestamp}] {message}\n")
        self.log_area.see("end")
        self.log_area.config(state="disabled")

    def check_env(self):
        if not os.path.exists(TOR_EXECUTABLE_PATH):
            self.log(f"CRITICAL ERROR: 'atw.exe' not found at {TOR_EXECUTABLE_PATH}")

    def force_kill_tor(self):
        try:
            subprocess.run(["taskkill", "/F", "/IM", "atw.exe"], stdout=subprocess.DEVNULL, stderr=subprocess.DEVNULL)
            subprocess.run(["taskkill", "/F", "/IM", "tor.exe"], stdout=subprocess.DEVNULL, stderr=subprocess.DEVNULL)
        except: pass

    def clean_lock(self, data_dir):
        lock = os.path.join(data_dir, "lock")
        if os.path.exists(lock):
            try: os.remove(lock) 
            except: pass

    def get_port_range(self):
        try:
            start = int(self.ports_var.get().strip())
        except:
            start = 9050
        
        try:
            limit = int(self.gen_limit_var.get().strip())
        except:
            limit = 50
            
        # Scalable up to 1000 as requested
        if limit < 1: limit = 1
        if limit > 2000: limit = 2000 # Increased hard limit
        
        return list(range(start, start + limit))

    def set_io_state(self, state):
        for w in self.widgets_to_lock:
            try:
                if isinstance(w, tk.Entry):
                    w.config(state=state)
                elif isinstance(w, ttk.Combobox):
                    w.config(state="readonly" if state=="normal" else "disabled")
                elif isinstance(w, ttk.Radiobutton):
                    w.state(['!disabled'] if state=="normal" else ['disabled'])
            except: pass

    def toggle_tor(self):
        if self.is_running: return
        threading.Thread(target=self.start_tor_thread, daemon=True).start()

    def start_tor_thread(self):
        self.start_btn.config(state="disabled")
        self.set_io_state("disabled") # LOCK UI
        
        self.status_text.config(text="INITIALIZING...", foreground="#D29922") # Gold
        self.status_dot.itemconfig(self.status_light, fill="#D29922")
        
        tor_path = TOR_EXECUTABLE_PATH
        data_dir = DATA_DIR_PATH
        
        self.force_kill_tor()
        time.sleep(0.5)
        
        if not os.path.exists(data_dir): os.makedirs(data_dir)
        self.clean_lock(data_dir)

        ports = self.get_port_range()
        if not ports:
            self.log("Error: No ports configured.")
            self.root.after(0, self.stop_tor)
            return

        cmd = [
            tor_path, "--DataDirectory", data_dir,
            "--GeoIPFile", GEOIP_FILE,
            "--GeoIPv6File", GEOIP6_FILE,
            "--KeepalivePeriod", "60", 
            "--NumEntryGuards", "3",
            "--SocksTimeout", "120" # Improve reliability
        ]

        self.log(f"Configuring {len(ports)} listeners. Please wait...")
        # Add ports efficiently
        for p in ports:
            cmd.extend(["--SocksPort", f"0.0.0.0:{p} IsolateSOCKSAuth"])

        # Mode & Rotation Logic
        mode = self.proxy_mode_var.get()
        if mode == "sticky":
             self.log("Mode: STICKY IP (Max Stability)")
             cmd.extend(["--MaxCircuitDirtiness", "7200"]) # 2 Hours
             cmd.extend(["--NewCircuitPeriod", "7200"])
        elif mode == "rotating_5m":
             self.log("Mode: ROTATING IP (5m Interval)")
             cmd.extend(["--MaxCircuitDirtiness", "300"])
             cmd.extend(["--NewCircuitPeriod", "300"])
        elif mode == "rotating_10m":
             self.log("Mode: ROTATING IP (10m Interval)")
             cmd.extend(["--MaxCircuitDirtiness", "600"])
             cmd.extend(["--NewCircuitPeriod", "600"])
        else:
             # Fallback
             self.log("Mode: ROTATING IP (Default 5m)")
             cmd.extend(["--MaxCircuitDirtiness", "300"])
             cmd.extend(["--NewCircuitPeriod", "300"])

        # Country Logic - Optimized
        loc_map = {
            "United States (US)": "{us}", "United Kingdom (GB)": "{gb}", "Canada (CA)": "{ca}",
            "Germany (DE)": "{de}", "France (FR)": "{fr}", "Netherlands (NL)": "{nl}",
            "Australia (AU)": "{au}", "Japan (JP)": "{jp}", 
            "Pakistan (PK)": "{pk}", "India (IN)": "{in}", 
            "Singapore (SG)": "{sg}", "Brazil (BR)": "{br}"
        }
        sel = self.country_var.get()
        if sel in loc_map:
            code = loc_map[sel]
            self.log(f"Routing: Exclusive Exit Nodes in {sel}")
            cmd.extend(["--ExitNodes", code, "--StrictNodes", "1"])
            
            # Speed Optimizations for US/General
            if "{us}" in code:
                # Prefer fast, stable nodes
                 cmd.extend(["--FastFirstHopPK", "0"]) # Reduce CPU load on Hopkins
            elif "{pk}" in code:
                self.log("Network: Optimized for Pakistan nodes.")
                cmd.extend(["--CircuitBuildTimeout", "120"]) 

        try:
            cf = subprocess.CREATE_NO_WINDOW if sys.platform == "win32" else 0
            
            self.log(f"Launching Phantom Core...")
            
            tor_dir = os.path.dirname(tor_path)
            
            self.tor_process = subprocess.Popen(
                cmd, 
                cwd=tor_dir, 
                stdout=subprocess.PIPE, 
                stderr=subprocess.STDOUT, 
                text=True, 
                bufsize=1, 
                universal_newlines=True, 
                creationflags=cf
            )
            self.is_running = True
            
            self.active_ports = ports 
            
            while self.is_running and self.tor_process.poll() is None:
                line = self.tor_process.stdout.readline()
                if not line: break
                l = line.lower()
                if "bootstrapped 100%" in l:
                    self.root.after(0, self.update_status_online)
                    self.root.after(0, lambda: self.log(">>> PHANTOM NETWORK SECURE (100%)"))
                elif "warn" in l or "err" in l:
                     if "socks" in l or "circuit" in l: # Filter spam
                        self.root.after(0, lambda x=line.strip(): self.log(x))
            
            if self.is_running:
                self.root.after(0, self.stop_tor)
                self.root.after(0, lambda: self.log("Engine stopped unexpectedly."))
        except Exception as e:
             self.log(f"Start Error: {e}")
             self.root.after(0, self.stop_tor)

    def update_status_online(self):
        self.status_text.config(text="PHANTOM ONLINE", foreground=ACCENT_COLOR)
        self.status_dot.itemconfig(self.status_light, fill=ACCENT_COLOR)
        self.stop_btn.config(state="normal")

    def stop_tor(self):
        self.is_running = False
        self.set_io_state("normal") # UNLOCK UI
        

        
        self.status_text.config(text="SYSTEM OFFLINE", foreground="#777")
        self.status_dot.itemconfig(self.status_light, fill="#555")
        self.start_btn.config(state="normal")
        self.stop_btn.config(state="disabled")
        
        self.log("Terminating Phantom Engine...")
        self.force_kill_tor() 
        self.tor_process = None

    def generate_proxies(self):
        if not self.is_running:
            self.log("Error: Phantom Engine must be ONLINE.")
            return
        
        if not hasattr(self, 'active_ports') or not self.active_ports:
            self.log("Error: No active ports found. Restart Engine.")
            return

        self.output_area.delete(1.0, "end")
        proxies = ""
        
        count = 0
        for port in self.active_ports:
            # Standard SOCKS5 Auth: username:password
            username = ''.join(random.choices(string.ascii_lowercase + string.digits, k=8))
            password = ''.join(random.choices(string.ascii_lowercase + string.digits, k=10))
            
            p = f"127.0.0.1:{port}:{username}:{password}"
            proxies += p + "\n"
            self.output_area.insert("end", p + "\n")
            count += 1
        
        self.log(f"Generated {count} unique identities.")
        self.last_gen = proxies

    def copy_clipboard(self):
        if hasattr(self, 'last_gen'):
            self.root.clipboard_clear()
            self.root.clipboard_append(self.last_gen)
            self.log("Copied to clipboard.")

    def on_close(self):
        self.force_kill_tor()
        self.root.destroy()
        sys.exit()

class PhantomSecurity:
    BASE_URL = "https://phantom.infozoomers.com"
    
    @staticmethod
    def _get_text(endpoint):
        try:
            url = f"{PhantomSecurity.BASE_URL}/{endpoint}"
            req = urllib.request.Request(url, headers={'User-Agent': 'Mozilla/5.0'})
            with urllib.request.urlopen(req, timeout=10) as response:
                return response.read().decode('utf-8').strip()
        except Exception as e:
            return None

    @staticmethod
    def check_maintenance():
        # "https://phantom.infozoomers.com/verify.php?check=maintenance"
        status = PhantomSecurity._get_text("verify.php?check=maintenance")
        return status.lower() == "true" if status else False

    @staticmethod
    def check_update():
        # "https://phantom.infozoomers.com/verify.php?check=update"
        status = PhantomSecurity._get_text("verify.php?check=update")
        return status.lower() == "true" if status else False

    @staticmethod
    def verify_key(key):
        # POST key to verify.php
        try:
            url = f"{PhantomSecurity.BASE_URL}/verify.php"
            data = urllib.parse.urlencode({'key': key}).encode()
            req = urllib.request.Request(url, data=data, headers={'User-Agent': 'Mozilla/5.0'})
            with urllib.request.urlopen(req, timeout=10) as response:
                result = response.read().decode('utf-8').strip()
                return result == "VALID"
        except Exception as e:
            return False

class LoginWindow:
    def __init__(self, root, on_success):
        self.root = root
        self.on_success = on_success
        self.root.title("PHANTOM SECURITY GATEWAY")
        self.root.geometry("500x650")
        self.root.configure(bg=BG_COLOR)
        self.root.overrideredirect(True) # Frameless look for premium feel
        
        # Center the window
        screen_width = self.root.winfo_screenwidth()
        screen_height = self.root.winfo_screenheight()
        x = (screen_width // 2) - (500 // 2)
        y = (screen_height // 2) - (650 // 2)
        self.root.geometry(f"500x650+{x}+{y}")

        self._draw_ui()
        self._run_initial_checks()

    def _draw_ui(self):
        # Main Frame with Border
        self.main_frame = tk.Frame(self.root, bg=BG_COLOR, highlightbackground=ACCENT_COLOR, highlightthickness=2)
        self.main_frame.pack(fill="both", expand=True)

        # Close Button (since frameless)
        close_btn = tk.Button(self.main_frame, text="✕", bg=BG_COLOR, fg=SECONDARY_TEXT, 
                              font=("Arial", 12), bd=0, activebackground=BG_COLOR, activeforeground="white",
                              command=lambda: sys.exit())
        close_btn.place(x=460, y=10)

        # Content Container
        content = tk.Frame(self.main_frame, bg=BG_COLOR)
        content.pack(expand=True, fill="x", padx=40)

        # Logo/Title
        tk.Label(content, text="AP", bg=BG_COLOR, fg=ACCENT_COLOR, font=("Impact", 60)).pack(pady=(0, 10))
        tk.Label(content, text="PHANTOM AUTHORIZATION", bg=BG_COLOR, fg="white", font=("Segoe UI", 14, "bold")).pack()
        
        self.status_lbl = tk.Label(content, text="INITIALIZING SECURITY PROTOCOLS...", bg=BG_COLOR, fg=SECONDARY_TEXT, font=("Consolas", 9))
        self.status_lbl.pack(pady=(30, 20))

        # Key Input Area (Hidden initially)
        self.input_frame = tk.Frame(content, bg=BG_COLOR)
        
        tk.Label(self.input_frame, text="ENTER LICENSE KEY", bg=BG_COLOR, fg=ACCENT_COLOR, font=("Segoe UI", 9, "bold")).pack(anchor="w", pady=(0,5))
        
        self.key_entry = tk.Entry(self.input_frame, bg=INPUT_BG, fg="white", insertbackground=ACCENT_COLOR, 
                                  font=("Consolas", 12), relief="flat", justify="center")
        self.key_entry.pack(fill="x", ipady=10, pady=(0, 20))
        
        # Custom Border for entry (Canvas line or Frame)
        tk.Frame(self.input_frame, bg=ACCENT_COLOR, height=1).pack(fill="x", pady=(0, 20))

        self.login_btn = tk.Button(self.input_frame, text="AUTHENTICATE ACCESS", bg=ACCENT_COLOR, fg="black",
                                   font=("Segoe UI", 11, "bold"), relief="flat", cursor="hand2", command=self.do_login)
        self.login_btn.pack(fill="x", ipady=10)

        # Update Area (Hidden initially)
        self.update_frame = tk.Frame(content, bg=BG_COLOR)
        tk.Label(self.update_frame, text="UPDATE REQUIRED", bg=BG_COLOR, fg=ALERT_COLOR, font=("Segoe UI", 16, "bold")).pack(pady=10)
        tk.Label(self.update_frame, text="Your version is obsolete.\nSecurity protocols require the latest version.", 
                 bg=BG_COLOR, fg=SECONDARY_TEXT, font=("Segoe UI", 10), justify="center").pack(pady=10)
        tk.Button(self.update_frame, text="DOWNLOAD UPDATE", bg=ACCENT_COLOR, fg="black", 
                  font=("Segoe UI", 11, "bold"), relief="flat", command=self.open_update).pack(fill="x", ipady=10, pady=20)

    def _run_initial_checks(self):
        # Run in thread to not freeze UI
        threading.Thread(target=self._check_logic, daemon=True).start()

    def _check_logic(self):
        # 1. Check Maintenance "on.txt"
        self.status_lbl.config(text="CHECKING SERVER STATUS...")
        try:
            if PhantomSecurity.check_maintenance():
                self.root.after(0, lambda: self._show_error("MAINTENANCE MODE", "The Bot is Disabled or Under Maintenance.\nPlease try again later."))
                return
        except:
             # If check fails, maybe allow or block? Creating safe fail: allow proceed to key check
             pass

        # 2. Check Update "update.txt"
        self.status_lbl.config(text="VERIFYING VERSION INTEGRITY...")
        try:
            if PhantomSecurity.check_update():
                self.root.after(0, self._show_update_ui)
                return
        except:
            pass

        # 3. Ready for Key
        self.root.after(0, self._show_login_ui)

    def _show_error(self, title, msg):
        self.status_lbl.config(text="ACCESS DENIED", fg=ALERT_COLOR)
        messagebox.showerror(title, msg)
        self.root.destroy()
        sys.exit()

    def _show_update_ui(self):
        self.status_lbl.pack_forget()
        self.update_frame.pack(fill="x", pady=20)

    def _show_login_ui(self):
        self.status_lbl.config(text="SECURE CONNECTION ESTABLISHED")
        self.input_frame.pack(fill="x", pady=10)
        self.key_entry.focus()

    def open_update(self):
        webbrowser.open("https://phantom.infozoomers.com")
        sys.exit()

    def do_login(self):
        key = self.key_entry.get().strip()
        if not key:
            messagebox.showwarning("Input Required", "Please enter your license key.")
            return
        
        self.login_btn.config(state="disabled", text="VERIFYING...")
        
        def verify_thread():
            if PhantomSecurity.verify_key(key):
                self.root.after(0, self.login_success)
            else:
                self.root.after(0, self.login_fail)
        
        threading.Thread(target=verify_thread, daemon=True).start()

    def login_success(self):
        # Do not destroy root here. It kills the loop.
        self.on_success()

    def login_fail(self):
        self.login_btn.config(state="normal", text="AUTHENTICATE ACCESS")
        messagebox.showerror("Authentication Failed", "Invalid License Key.\nAccess Denied.")

if __name__ == "__main__":
    try:
        from ctypes import windll
        windll.shcore.SetProcessDpiAwareness(1)
    except: pass
    
    root = tk.Tk()
    
    # Logic to switch: Start Login first
    # We hide the root initially or just use it for LoginWindow if we want to reuse it?
    # Better: Use root for LoginWindow, destroy it, create new root for App? 
    # Tkinter doesn't like creating new root after destroying old one sometimes.
    # Approach: Use root for Login, clear it, then build App on same root.
    
    # Actually, simpler: Use root for Login. If success, destroy Login widgets (or the frame), resize root, and build App.
    # But LoginWindow makes root frameless (overrideredirect). App needs frame.
    
    def start_app():
        try:
            # Restore root window to standard state
            root.overrideredirect(False)
            
            # Clear everything
            for widget in root.winfo_children():
                widget.destroy()
            
            # Initialize Main App
            app = PhantomProxyBot(root)
            root.protocol("WM_DELETE_WINDOW", app.on_close)
            root.app_instance = app # Keep reference
            
            # Force update to ensure frame is drawn
            root.update()
        except Exception as e:
            messagebox.showerror("Critical Error", f"Failed to start app: {e}")
            sys.exit(1)
        
    # Hack for LoginWindow: It treats root as its window.
    # We need to set up LoginWindow to use root.
    
    login = LoginWindow(root, start_app)
    
    root.mainloop()
